﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Text;
using System.Linq;
using System.Numerics;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

namespace graphicbox2d
{
    /// <summary>
    /// ポリゴン図形クラス
    /// </summary>
    public class Polygon2D : Object2D, ILineProperty
    {
        // ===============================================================================
        // 公開プロパティ
        // ===============================================================================

        /// <summary>
        /// 図形の種類
        /// </summary>
        public override eObject2DType m_Type => eObject2DType.Polygon;

        /// <summary>
        /// ポリゴンの頂点リスト
        /// </summary>
        public List<PointF> Points { get; set; } = new List<PointF>();

        /// <summary>
        /// 線の太さ
        /// </summary>
        public int Width { get; set; }

        /// <summary>
        /// 線の種類
        /// </summary>
        public DashStyle LineStyle { get; set; }

        /// <summary>
        /// 塗りつぶしフラグ
        /// </summary>
        public bool IsFilled { get; set; }

        // ===============================================================================
        // 非公開プロパティ
        // ===============================================================================

        /// <summary>
        /// 図形の中心点
        /// </summary>
        internal override Vector2 CenterPoint { get { return GetCenterPoint(); } }

        /// <summary>
        /// ポリゴン図形を完全内包する円（外接円）の半径
        /// </summary>
        internal float CircumCircleR { get { return GetCircumCircleR(); } }


        // ===============================================================================
        // 公開メソッド
        // ===============================================================================

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Polygon2D()
        {
            Points = new List<PointF>();

            Width = 1;
            LineStyle = DashStyle.Solid;
            IsFilled = false;
            IsSelect = false;
        }

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~Polygon2D()
        {
            Points.Clear();
        }

        /// <summary>
        /// コピーを作成する
        /// </summary>
        /// <returns>コピーしたオブジェクト</returns>
        public override Object2D Clone()
        {
            Polygon2D polygon = new Polygon2D();

            // 基底クラスのデータをコピーする
            this.BaseCopyDataTo(polygon);

            // ポリゴン固有のデータをコピーする
            polygon.Width = this.Width;
            polygon.IsFilled = this.IsFilled;
            polygon.IsSelect = this.IsSelect;

            foreach (var pt in this.Points)
            {
                polygon.Points.Add(new PointF(pt.X, pt.Y));
            }

            return polygon;
        }

        // ===============================================================================
        // 非公開メソッド
        // ===============================================================================

        /// <summary>
        /// ポリゴン座標無しでコピーを作成する
        /// </summary>
        /// <returns>コピーしたオブジェクト</returns>
        internal Object2D CloneWithoutPoints()
        {
            Polygon2D polygon = new Polygon2D();

            // 基底クラスのデータをコピーする
            this.BaseCopyDataTo(polygon);

            // ポリゴン固有のデータをコピーする
            polygon.Width = this.Width;
            polygon.IsFilled = this.IsFilled;
            polygon.IsSelect = this.IsSelect;

            return polygon;
        }

        /// <summary>
        /// マウスヒット中の図形（拡大した図形）を返す。
        /// </summary>
        /// <returns>拡大された図形</returns>
        internal override Object2D GetHitObject()
        {
            List<PointF>  OutPoints;
            GraphicCaluculate.GetScalingPolygon(this.Points, CenterPoint.ToPointF(), MouseHitPolyOffset, out OutPoints);

            // ポイントを除いたポリゴンをコピーする
            Polygon2D out_polygon = (Polygon2D)this.CloneWithoutPoints();

            // 拡大したポイントを設定する
            out_polygon.Points.AddRange(OutPoints);

            return out_polygon;
        }

        /// <summary>
        /// マウスポイントがこの図形にヒットしているか判定する。
        /// </summary>
        /// <param name="MousePoint">マウスポイント</param>
        /// <param name="MusePointRange">マウスヒット半径</param>
        /// <returns></returns>
        internal override eMouseHitType IsHitMousePoint(PointF MousePoint, float MusePointRange)
        {
            eMouseHitType eMouseHitType;

            Vector2 _CenterPoint;
            float   _CircumCircleR;

            GetCenterPointAndCircumCircleR(out _CenterPoint, out _CircumCircleR);

            if (IsFilled == true)
            {
                eMouseHitType = GraphicCaluculate.IsHitMouseRangeFillPolygon(_CenterPoint.ToPoint(), _CircumCircleR, Points, MousePoint, MusePointRange);
            }
            else
            {
                eMouseHitType = GraphicCaluculate.IsHitMouseRangeLinePolygon(_CenterPoint.ToPoint(), _CircumCircleR, Points, Width, MousePoint, MusePointRange);
            }

            return eMouseHitType;
        }

        /// <summary>
        /// マウスポイントとこの図形の距離を取得する
        /// </summary>
        /// <param name="X">マウスポイントX座標</param>
        /// <param name="Y">マウスポイントY座標</param>
        /// <returns>距離</returns>
        internal override float GetDistanceHitMousePoint(float X, float Y)
        {
            Vector2 MousePoint = new Vector2(X, Y);

            return Vector2.Distance(MousePoint, CenterPoint);
        }

        /// <summary>
        /// 図形を移動させる
        /// </summary>
        /// <param name="Movement">移動量</param>
        internal override void Move(PointF Movement)
        {
            for (int i = 0; i < Points.Count; i++)
            {
                Points[i] = new PointF(Points[i].X + Movement.X, Points[i].Y + Movement.Y);
            }
        }

        /// <summary>
        /// 図形を移動させる
        /// </summary>
        /// <param name="X">移動量X</param>
        /// <param name="Y">移動量Y</param>
        internal override void Move(float X, float Y)
        {
            for (int i = 0; i < Points.Count; i++)
            {
                Points[i] = new PointF(Points[i].X + X, Points[i].Y + Y);
            }
        }

        /// <summary>
        /// バウンディングボックスを取得する
        /// </summary>
        internal override PointF[] GetBoundingBox()
        {
            return GraphicCaluculate.GetBoundingBoxPolygon(Points.ToArray());
        }

        /// <summary>
        /// ポリゴンの中心点と外接円の半径を同時に取得する
        /// </summary>
        /// <param name="centerPoint">ポリゴンの中心点</param>
        /// <param name="circumCircleR">外接円の半径</param>
        internal void GetCenterPointAndCircumCircleR(out Vector2 centerPoint, out float circumCircleR)
        {
            centerPoint = GraphicCaluculate.CaluculateCenterPoint(Points);
            circumCircleR = GraphicCaluculate.CaluculateCircumCircleR(centerPoint, Points);
        }

        /// <summary>
        /// 中心点を取得する
        /// </summary>
        /// <returns></returns>
        internal Vector2 GetCenterPoint()
        {
            return GraphicCaluculate.CaluculateCenterPoint(Points);
        }

        /// <summary>
        /// 外接円を取得する
        /// </summary>
        /// <returns></returns>
        internal float GetCircumCircleR()
        {
            Vector2 centerPoint = GraphicCaluculate.CaluculateCenterPoint(Points);

            return GraphicCaluculate.CaluculateCircumCircleR(centerPoint, Points);
        }
    }
}
